clear
program drop _all
set mem 700m
cd "C:\Users\Andrew\My publications\Ethnic ghettos\"
set more off

/*

****************************************************
****************************************************
* This first part of the dofile collapses the IPUMS files down to something usable.
****************************************************
****************************************************

***************************************************
* Conversion from DAT to Stata format
* Note that this uses an IPUMS data extract which is men only
***************************************************
program define ipumsinfix
label var year "Census year"
label var statefip "State (FIPS code)"
label var gq "Group quarters status"
label var perwt "Person weight"
label var age "Age"
label var sex "Sex"
label var bpld "Birthplace -- Detailed"
label var ancest1g "Ancestry, first response -- General"
label var yrimmig "Year of immigration"
label var educrec "Educational attainment recode"
label var empstatd "Employment status -- Detailed"
label var wkswork1 "Weeks worked last year"
label var uhrswork "Usual hours worked per week"
label var incwage "Wage and salary income"
compress
end
* This runs the above program. It does it year-by-year, for computational speed.
for any 98 99 00 \ any 1980 1990 2000: clear \ infix using ghettos.dct if year==X \ ipumsinfix \ save temp_Y.dta, replace

******************************************
* Recoding
* ipumscollapse1 collapses all US. 
* ipumscollapse2 collapses by census region
******************************************

program define ipumscollapse1
replace year=(year*10)+1000
recode year 1000=2000 1010=2001 1020=2002 1030=2003
replace yrimmig=yrimmig+1000
recode yrimmig 1000=. 1001=2000
* Recoding age into 5-year bands
replace age=5*(int(age/5))
* Collapsing education into 4 categories (less than HS, HS only, some coll, college+)
recode educrec 0/6=1 7=2 8=3 9=4
* Recoding birth country - collapsing all US
recode bpld 1/12092=1 90000/90010=. 90011=1 90020=. 90021=1 95000/max=.
* Recoding employment status
recode empstatd 0=. 10/15=1 20/34=0
* Generating annual hours
gen ann_hours=wkswork1*uhrswork
replace incwage=. if incwage==999999
* Generating hourly wages
gen hwage=incwage/ann_hours
* Collapsing
gen region_code=bpld
recode region_code 1=1 20000=20 20001/21090=21 25000/29998=22 29999/30090=29 30091/40999=30 41000/41899=40 41901/42999=41 42901/44999=42 44001/45999=44 45901/49899=45 49901/50999=49 50901/51999=50 51911/52999=51 54800=51 52401/54799=52 54900=52 59901/69999=59 60100/70999=60 *=.
* Cleaning up the data
for any educrec age region year: drop if X==.
sum 
collapse yrimmig empstatd ann_hours incwage hwage (count) n=empstatd [aw=perwt], by(year educrec age region_code)
sort year
end

program define ipumscollapse2
* Generating a census division variable
gen census_division=.
for num 01 02 04 05 06 08 09 10 11 12 13 15 16 17 18 19 20 21 22 23 24 25 26 27 28 29 30 31 32 33 34 35 36 37 38 39 40 41 42 44 45 46 47 48 49 50 51 53 54 55 56 \ num 6 9 8 7 9 8 1 5 5 5 5 9 8 3 3 4 4 6 7 1 5 1 3 4 6 4 8 4 8 1 2 8 2 5 4 3 7 9 2 1 5 4 6 7 8 1 5 9 5 3 8: replace census_division=Y if statefip==X
for num 61 62 63 64 65 66 67 68  \ num 1 1 4 5 8 8 8 9 : replace census_division=Y if statefip==X
for num 94 97 99 : replace census_division=. if statefip==X
replace year=(year*10)+1000
recode year 1000=2000 1010=2001 1020=2002 1030=2003
replace yrimmig=yrimmig+1000
recode yrimmig 1000=. 1001=2000
* Recoding age into 5-year bands
replace age=5*(int(age/5))
* Collapsing education into 4 categories (less than HS, HS only, some coll, college+)
recode educrec 0/6=1 7=2 8=3 9=4
* Recoding birth country - collapsing all US
recode bpld 1/12092=1 90000/90010=. 90011=1 90020=. 90021=1 95000/max=.
* Recoding employment status
recode empstatd 0=. 10/15=1 20/34=0
* Generating annual hours
gen ann_hours=wkswork1*uhrswork
replace incwage=. if incwage==999999
* Generating hourly wages
gen hwage=incwage/ann_hours
* Collapsing
gen region_code=bpld
recode region_code 1=1 20000=20 20001/21090=21 25000/29998=22 29999/30090=29 30091/40999=30 41000/41899=40 41901/42999=41 42901/44999=42 44001/45999=44 45901/49899=45 49901/50999=49 50901/51999=50 51911/52999=51 54800=51 52401/54799=52 54900=52 59901/69999=59 60100/70999=60 *=.
* Cleaning up the data
for any educrec age region year: drop if X==.
sum 
collapse yrimmig empstatd ann_hours incwage hwage (count) n=empstatd [aw=perwt], by(year educrec age region_code census_division)
sort year
end

***************************************************
* This part runs the programs above
***************************************************
* Collapse - no census regions
for Y in any 1980 1990 2000: use temp_Y.dta, clear \ ipumscollapse1 \ save ghettos_1_Y, replace 

* Collapse - with census regions
for Y in any 1980 1990 2000: use temp_Y.dta, clear \ ipumscollapse2 \ save ghettos_2_Y, replace 

***************************************************
* This takes the 3 US-level ethnicity files, combines them, 
* and does some recoding.
* 
* NB. It starts from a file containing the variables: 
* -- yrimmig empstatd ann_hours incwage hwage n
* Collapsed by:
* -- year educrec age region_code 
***************************************************

* Combining files
use ghettos_1_1980.dta, clear
for num 1990 2000: sort year \ merge year using ghettos_1_X \ drop _merge

* Age restriction
drop if age<25 | age>60

* Creating LHS variables relative to natives of same {year educrec age}
for any empstatd ann_hours incwage hwage: gen temp=X if region==1 \ bysort year educrec age: egen n_X=max(temp) \ drop temp \ gen r_X=X/n_X if region~=1
for any empstatd ann_hours incwage hwage: la var r_X "Ratio of immigrant X to native X"

* Weighting all censuses equally
bysort year: egen temp1=sum(n)
egen temp2=sum(n)
replace n=n/((temp1/temp2)*4)
drop temp1 temp2

* Creating cell weights
bysort year: egen temp1=sum(n)
bysort region year: egen temp2=sum(n)
gen groupsize=(temp2-n)/(temp1-n)

* Creating variables to use in the regressions
gen birthyear=year-age
gen ysi=year-yrimmig
gen ysi2=ysi^2
gen temp3=ysi*n
bysort region year: egen temp4=sum(temp3)
gen mean_ysi=(temp4-temp3)/(temp2-n)
drop temp*
egen pseudo_person=group(educrec birthyear region)
la var ysi "Years Since Immigration"
la var ysi2 "Years Since Immigration Squared"

save deadimmig_notbyregion.dta, replace


***************************************************
* This takes the 3 census division-level ethnicity files, combines them, 
* and does some recoding.
* 
* NB. It starts from a file containing the variables: 
* -- yrimmig empstatd ann_hours incwage hwage n
* Collapsed by:
* -- year educrec age region_code census_division
***************************************************

* Combining files
use ghettos_2_1980.dta, clear
for num 1990 2000: sort year \ merge year using ghettos_2_X \ drop _merge

* Age restriction
drop if age<25 | age>60

* Creating LHS variables relative to natives of same {census_division year educrec age}
for any empstatd ann_hours incwage hwage: gen temp=X if region==1 \ bysort census_division year educrec age: egen n_X=max(temp) \ drop temp \ gen r_X=X/n_X if region~=1
for any empstatd ann_hours incwage hwage: la var r_X "Ratio of immigrant X to native X"

* Weighting all censuses equally
bysort year: egen temp1=sum(n)
egen temp2=sum(n)
replace n=n/((temp1/temp2)*4)
drop temp1 temp2

* Creating cell weights
bysort census_division year: egen temp1=sum(n)
bysort census_division region year: egen temp2=sum(n)
gen groupsize=(temp2-n)/(temp1-n)

* Creating variables to use in the regressions
gen birthyear=year-age
gen ysi=year-yrimmig
gen ysi2=ysi^2
gen temp3=ysi*n
bysort census_division region year: egen temp4=sum(temp3)
gen mean_ysi=(temp4-temp3)/(temp2-n)
drop temp*
egen pseudo_person=group(census_division educrec birthyear region)
la var ysi "Years Since Immigration"
la var ysi2 "Years Since Immigration Squared"

save deadimmig_byregion.dta, replace

*/

****************************************************
****************************************************
* This part of the dofile produces Tables 3, 4 & 5 in the paper
* (In an earlier version, it was called 'smalldata.do')
****************************************************
****************************************************
clear
set mem 50m
version 10.0
set more off
*cd "C:\Users\Tim\Documents\Andrew\"
cd "C:\Users\Andrew\My publications\Ethnic ghettos\Data & dofiles (Mar 2009)\"

use deadimmig_notbyregion, clear
gen link=region*10000+year

joinby link using linkdata
gen stknowsq=stknow*stknow
gen lngdp10 = ln( gdp10ago)
gen lngdp20 = ln( gdp20ago)
gen lndist=ln(distance)
gen lnempstat= ln(r_empstatd)
gen stknow0 = stknow/10
gen stknow0sq = stknow0*stknow0
gen ysi0 = ysi/10
gen ysi00sq =ysi0*ysi0
gen meanysi0 = mean_ysi/10
gen paststk0 = paststkave/10

gen edu4=1 if educrec==4
gen lnincwage=ln(r_incwage)
gen lnhwage=ln(r_hwage)
gen lnannhours=ln(r_ann_hours)
gen w=n /* Weighting by the number of migrants in the cell */

replace edu4 = 0 if edu4 ==. 
gen paststedu4 = paststk0*edu4
gen ancestedu4 = ancestry*edu4
gen stk30ago0 = stk30ago/10
gen stk70ago0 = stk70ago/10
gen stk120ago0 = stk120ago/10
gen stk30edu4 = stk30ago*edu4/10
gen stk70edu4 = stk70ago*edu4/10
gen stk120edu4 = stk120ago*edu4/10

log using smallresults.log, replace

* The referee asked us to: 
* Cluster SE at the ethnicity group level in tables 3 and 5
* Cluster SE at the ethnicity*time level in tables 4, 6, and 7

* Table 3
xi: reg  lnincwage ysi0 ysi00sq meanysi0 stknow0 stknow0sq paststk0 paststedu4 gdp10ago edu10ago gini10ago lndist i.educrec i.age i.year if year>1970 [aw=w], cl(region_code)
outreg using table345.doc, coefastr nocons bracket 3aster replace bdec(3) se ct("Table 3")
xi: reg  lnhwage ysi0 ysi00sq meanysi0 stknow0 stknow0sq paststk0 paststedu4 gdp10ago edu10ago gini10ago lndist i.educrec i.age i.year if year>1970 [aw=w], cl(region_code)
outreg using table345.doc, coefastr nocons bracket 3aster append bdec(3) se ct("Table 3")
xi: reg  lnannhours ysi0 ysi00sq meanysi0 stknow0 stknow0sq paststk0 paststedu4 gdp10ago edu10ago gini10ago lndist i.educrec i.age i.year if year>1970 [aw=w], cl(region_code)
outreg using table345.doc, coefastr nocons bracket 3aster append bdec(3) se ct("Table 3")

* Table 4 - weighted & clustered
* Note that the standard <xtreg,re> command doesn't like weighting, and doesn't cluster when the clusters vary within panels. We could use xtregre2 if we only wanted weights, but that command won't cluster.
* So the way we do it is to expand the dataset by a factor of int(w/20). On average, this expands the dataset by a factor of 21. This approximately takes care of the weighting issue. 
* We then deal with the clustering issue by bootstrapping, using the regular <xtreg,re> command.
* Having run the bootstrapped regression on the expanded dataset, we restore the dataset to its previous form.
preserve
gen expand=int(w/20)
recode expand 0=1
expand expand
xi: bootstrap,cluster(link): xtreg lnincwage ysi0 ysi00sq meanysi0 stknow0 stknow0sq paststk0 paststedu4 gdp10ago edu10ago gini10ago i.educrec i.age i.year if year>1970, i(region_code) re
outreg using table345.doc, coefastr nocons bracket 3aster append bdec(3) se ct("Table 4 - income RE, weighted clustered")
restore
xi: areg lnincwage ysi0 ysi00sq meanysi0 stknow0 stknow0sq paststk0 paststedu4 gdp10ago edu10ago gini10ago i.educrec i.age i.year if year>1970 [aw=w], a(region_code) cl(link)
outreg using table345.doc, coefastr nocons bracket 3aster append bdec(3) se ct("Table 4 - income FE, weighted clustered")
preserve
gen expand=int(w/20)
recode expand 0=1
expand expand
xi: bootstrap,cluster(link): xtreg lnhwage ysi0 ysi00sq meanysi0 stknow0 stknow0sq paststk0 paststedu4 gdp10ago edu10ago gini10ago i.educrec i.age i.year if year>1970, i(region_code) re
outreg using table345.doc, coefastr nocons bracket 3aster append bdec(3) se ct("Table 4 - wage RE, weighted clustered")
restore
xi: areg lnhwage ysi0 ysi00sq meanysi0 stknow0 stknow0sq paststk0 paststedu4 gdp10ago edu10ago gini10ago i.educrec i.age i.year if year>1970 [aw=w], a(region_code) cl(link)
outreg using table345.doc, coefastr nocons bracket 3aster append bdec(3) se ct("Table 4 - wage FE, weighted clustered")

* Table 5
xi: reg lnincwage ysi0 ysi00sq meanysi0 stknow0 stknow0sq stk120ago0 stk120edu4 gdp10ago edu10ago gini10ago lndist i.educrec i.age i.year if year>1970 [aweight=w], robust cl(region_code)
outreg using table345.doc, coefastr nocons bracket 3aster append bdec(3) se ct("Table 5")
xi: reg lnincwage ysi0 ysi00sq meanysi0 stknow0 stknow0sq stk70ago0 stk70edu4 gdp10ago edu10ago gini10ago lndist i.educrec i.age i.year if year>1970 [aweight=w],robust cl(region_code)
outreg using table345.doc, coefastr nocons bracket 3aster append bdec(3) se ct("Table 5")
xi: reg lnincwage ysi0 ysi00sq meanysi0 stknow0 stknow0sq stk30ago0 stk30edu4 gdp10ago edu10ago gini10ago lndist i.educrec i.age i.year if year>1970 [aweight=w], robust cl(region_code)
outreg using table345.doc, coefastr nocons bracket 3aster append bdec(3) se ct("Table 5")
xi: ivreg lnincwage ysi0 ysi00sq meanysi0 stknow0 stknow0sq (ancestry ancestedu4 =paststk0 paststedu4)  gdp10ago edu10ago gini10ago lndist i.educrec i.age i.year if year>1970 [aweight=w], robust cl(region_code)
outreg using table345.doc, coefastr nocons bracket 3aster append bdec(3) se ct("Table 5")

* Hausman tests for Table 4
for any lnincwage lnhwage: xi: xtreg X ysi0 ysi00sq meanysi0 stknow0 stknow0sq paststk0 paststedu4 gdp10ago edu10ago gini10ago i.educrec i.age i.year if year>1970, i(region_code) fe \ estimates store fixed \ xi: xtreg X ysi0 ysi00sq meanysi0 stknow0 stknow0sq paststk0 paststedu4 gdp10ago edu10ago gini10ago i.educrec i.age i.year if year>1970, i(region_code) re \ hausman fixed .

* xtreg,fe approach (not clustered, not weighted) to get the 3 R-squared statistics in Table 4
for any lnincwage lnhwage: xi: xtreg X ysi0 ysi00sq meanysi0 stknow0 stknow0sq paststk0 paststedu4 gdp10ago edu10ago gini10ago i.educrec i.age i.year if year>1970, fe i(region_code) 

******************************
* Additional robustness checks
******************************
* For comparison, we re-run the OLS annual earnings estimate (table 3, col 1)
xi: reg lnincwage ysi0 ysi00sq meanysi0 stknow0 stknow0sq paststk0 paststedu4 gdp10ago edu10ago gini10ago lndist i.educrec i.age i.year if year>1970 [aw=w], cl(region_code)
outreg using table345.doc, coefastr nocons bracket 3aster append bdec(3) se ct("Robustness - same as table 3 col 1")
egen educrec_age_year=group(educrec age year)
xi: areg lnincwage ysi0 ysi00sq meanysi0 stknow0 stknow0sq paststk0 paststedu4 gdp10ago edu10ago gini10ago lndist if year>1970 [aw=w], cl(region_code) a(educrec_age_year)
outreg using table345.doc, coefastr nocons bracket 3aster append bdec(3) se ct("Robustness")
for any stknow0 paststk0 : gen lnX=ln(X) \ gen lnXsq=(ln(X))^2
gen lnpaststedu4 =lnpaststk0*edu4
xi: reg lnincwage ysi0 ysi00sq meanysi0 lnstknow0 lnpaststk0 lnpaststedu4 gdp10ago edu10ago gini10ago lndist i.educrec i.age i.year if year>1970 [aw=w], cl(region_code)
outreg using table345.doc, coefastr nocons bracket 3aster append bdec(3) se ct("Robustness")
xi: reg lnincwage ysi0 ysi00sq meanysi0 lnstknow0 lnstknow0sq lnpaststk0 lnpaststk0sq lnpaststedu4 gdp10ago edu10ago gini10ago lndist i.educrec i.age i.year if year>1970 [aw=w], cl(region_code)
outreg using table345.doc, coefastr nocons bracket 3aster append bdec(3) se ct("Robustness")

* Without the education interaction
xi: reg lnincwage ysi0 ysi00sq meanysi0 stknow0 stknow0sq paststk0 gdp10ago edu10ago gini10ago lndist i.educrec i.age i.year if year>1970 [aw=w], cl(region_code)
outreg using table345.doc, coefastr nocons bracket 3aster append bdec(3) se ct("Robustness - no educ interaction")
xi: areg lnincwage ysi0 ysi00sq meanysi0 stknow0 stknow0sq paststk0 paststedu4 gdp10ago edu10ago gini10ago i.educrec i.age i.year if year>1970 [aw=w], a(region_code) cl(link)
outreg using table345.doc, coefastr nocons bracket 3aster append bdec(3) se ct("Robustness - same as table 4 col 2")
xi: areg lnincwage ysi0 ysi00sq meanysi0 stknow0 stknow0sq paststk0 gdp10ago edu10ago gini10ago i.educrec i.age i.year if year>1970 [aw=w], a(region_code) cl(link)
outreg using table345.doc, coefastr nocons bracket 3aster append bdec(3) se ct("Robustness - no educ interaction")

log close

******************************
* Figure 1: Ratio of to native-born of earnings for males 
* aged 25-64 for 16 origin groups and three census years plotted 
* against the ratio of the average share of population in the 
* over the previous 120 years to the current share of population.
******************************
collapse r_incwage paststk0 stknow0 [w=w],by(region_code year)
gen stockratio=paststk0/stknow0
gen region_name=""
for num 20 21 22 29 30 40 41 42 44 45 49 50 51 52 59 60 \ any "Mexico" "Central America" "Caribbean" "South America" "Scandinavia" "UK & Ireland" "Western Europe" "Southern Europe" "Central/Eastern Europe" "Russian Empire" "East Asia" "Southeast Asia" "India/Southwest Asia" "Middle East" "Africa" "Australia & NZ": replace region_name="Y" if region_code==X
gen yearstring=string(year)
gen str long_region_name=region_name+"-"+yearstring
gen str short_region_name=""
for any "Mexico" "Central America" "Caribbean" "South America" "Scandinavia" "UK & Ireland" "Western Europe" "Southern Europe" "Central/Eastern Europe" "Russian Empire" "East Asia" "Southeast Asia" "India/Southwest Asia" "Middle East" "Africa" "Australia & NZ" \ any MEX CAM CAR SA SCA UK WE SE CE RUS EA SEA SWA ME AF AUS: replace short_region_name="Y" if region_name=="X"
replace short_region_name=short_region_name+substr(yearstring,3,2)
set scheme s1mono
tw scatter r_incwage stockratio,mlabel(short_region_name) || lfit r_incwage stockratio, ti("Figure 1: Immigrant Relative Earnings and" "Past/Present Immigrant Stock, 1980-2000") yti("Relative annual earnings") xti("Past stock/present stock") legend(off)
* Each year separately:
for any 1980(10)2000: tw scatter r_incwage stockratio if year==X,mlabel(short_region_name) || lfit r_incwage stockratio if year==X, ti("Figure 1: Immigrant Relative Earnings and" "Past/Present Immigrant Stock, X") yti("Relative annual earnings") xti("Past stock/present stock") legend(off)

****************************************************
****************************************************
* This part of the dofile produces Tables 6 & 7 in the paper
* (In an earlier version, it was called 'earnreg.do')
****************************************************
****************************************************

version 10.0
set more off
*cd "C:\Documents and Settings\hatton\My Documents\AndrewUpdate\"
cd "C:\Users\Andrew\My publications\Ethnic ghettos\Data & dofiles (Mar 2009)\"

use deadimmig_byregion, clear

gen origin=1 if region_code==20
replace origin=2 if region_code==21
replace origin=3 if region_code==22
replace origin=4 if region_code==29
replace origin=5 if region_code==30
replace origin=6 if region_code==40
replace origin=7 if region_code==41
replace origin=8 if region_code==42
replace origin=9 if region_code==44
replace origin=10 if region_code==45
replace origin=11 if region_code==49
replace origin=12 if region_code==50
replace origin=13 if region_code==51
replace origin=14 if region_code==52
replace origin=15 if region_code==59
replace origin=16 if region_code==60
tabulate origin

gen weight=n /* Weighting by the number of migrants in the cell */
egen region_year=group(region_code year)

gen linkvar= (1000000*census_division)+(10000*origin)+year
joinby linkvar using reglink.dta
gen stknow0 =stknow/10
gen stknow0sq =stknow0*stknow0
gen ysi0 = ysi/10
gen ysi00sq =ysi0*ysi0
gen meanysi0 = mean_ysi/10
gen paststk0 = paststkave/10
gen edu4=1 if educrec==4
replace edu4 = 0 if edu4==.
gen paststedu4 = paststk0*edu4
gen ancestedu4= ancestry*edu4
gen stk30ago0 = stk30ago/10
gen stk70ago0 = stk70ago/10
gen stk120ago0 = stk120ago/10
gen stk30edu4 = stk30ago*edu4/10
gen stk70edu4 = stk70ago*edu4/10
gen stk120edu4 = stk120ago*edu4/10

gen lnhours = ln(r_ann_hours) 
gen lnincome = ln(r_incwage)
gen lnhwage = ln(r_hwage)
gen lngdp10 = ln(gdpratio10ago)
gen lngdp20 = ln(gdpratio20ago)
gen lndist = ln(distance)
gen lnempstat = ln(r_empstatd)
gen lnregdist = ln(regdist)
log using regionresults, replace

* The referee asked us to: 
* Cluster SE at the ethnicity group level in tables 3 and 5
* Cluster SE at the ethnicity*time level in tables 4, 6, and 7

* Table 6
xi: reg lnincome ysi0 ysi00sq meanysi0 stknow0 stknow0sq paststk0 paststedu4 lnregdist i.educrec i.age i.origin i.census_division i.year if year!=1970 [aw=weight], robust cl(region_year)
outreg using table67.doc, coefastr nocons bracket 3aster replace bdec(3) se ct("Table 6")
xi: reg lnhwage ysi0 ysi00sq meanysi0 stknow0 stknow0sq paststk0 paststedu4 lnregdist i.educrec i.age  i.origin i.census_division i.year if year!=1970 [aw=weight], robust cl(region_year)
outreg using table67.doc, coefastr nocons bracket 3aster append bdec(3) se ct("Table 6")
xi: reg lnincome ysi0 ysi00sq meanysi0 stknow0 stknow0sq paststk0 paststedu4 gdpratio10ago edu10ago gini10ago lnregdist i.educrec i.age  i.census_division i.year if year!=1970 [aw=weight], robust cl(region_year)
outreg using table67.doc, coefastr nocons bracket 3aster append bdec(3) se ct("Table 6")
xi: reg lnhwage ysi0 ysi00sq meanysi0 stknow0 stknow0sq paststk0 paststedu4 gdpratio10ago edu10ago gini10ago lnregdist i.educrec i.age  i.census_division i.year if year!=1970 [aw=weight], robust cl(region_year)
outreg using table67.doc, coefastr nocons bracket 3aster append bdec(3) se ct("Table 6")

xi: reg lnincome ysi0 ysi00sq meanysi0 stknow0 stknow0sq stk120ago0 stk120edu4 lnregdist i.educrec i.age i.origin i.census_division i.year if year!=1970 [aw=weight], robust cl(region_year)
outreg using table67.doc, coefastr nocons bracket 3aster append bdec(3) se ct("Table 7")
xi: reg lnincome ysi0 ysi00sq meanysi0 stknow0 stknow0sq stk70ago0 stk70edu4 lnregdist i.educrec i.age i.origin i.census_division i.year if year!=1970 [aw=weight], robust cl(region_year)
outreg using table67.doc, coefastr nocons bracket 3aster append bdec(3) se ct("Table 7")
xi: reg lnincome ysi0 ysi00sq meanysi0 stknow0 stknow0sq stk30ago0 stk30edu4 lnregdist i.educrec i.age i.origin i.census_division i.year if year!=1970 [aw=weight], robust cl(region_year)
outreg using table67.doc, coefastr nocons bracket 3aster append bdec(3) se ct("Table 7")
xi: ivreg lnincome ysi0 ysi00sq meanysi0 stknow0 stknow0sq (ancestry ancestedu4 = paststk0 paststedu4) lnregdist i.educrec i.age i.origin i.census_division i.year if year!=1970 [aw=weight], robust cl(region_year)
outreg using table67.doc, coefastr nocons bracket 3aster append bdec(3) se ct("Table 7")

log close